<?php
namespace CmsmastersElementor\Modules\AdvancedTitle\Widgets;

use CmsmastersElementor\Base\Base_Widget;
use CmsmastersElementor\Controls\Groups\Group_Control_Vars_Text_Background;
use CmsmastersElementor\Controls_Manager as CmsmastersControls;
use CmsmastersElementor\Modules\Settings\Kit_Globals;
use CmsmastersElementor\Utils as CmsmastersUtils;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Icons_Manager;
use Elementor\Plugin;
use Elementor\Repeater;
use Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Advanced_Title extends Base_Widget {

	/**
	 * Get widget title.
	 *
	 * Retrieve the widget title.
	 *
	 * @since 1.20.0
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Advanced Title', 'cmsmasters-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @since 1.20.0
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'cmsicon-advanced-title';
	}

	/**
	 * Get widget unique keywords.
	 *
	 * Retrieve the list of unique keywords the widget belongs to.
	 *
	 * @since 1.20.0
	 *
	 * @return array Widget unique keywords.
	 */
	public function get_unique_keywords() {
		return array(
			'advanced',
			'title',
		);
	}

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the widget requires.
	 *
	 * @since 1.20.0
	 *
	 * @return array Widget style dependencies.
	 */
	public function get_style_depends(): array {
		return array( 'widget-cmsmasters-advanced-title' );
	}

	/**
	 * Outputs elementor widget container to the frontend if `Optimized Markup` is enabled.
	 *
	 * @since 1.20.0
	 */
	public function has_widget_inner_wrapper(): bool {
		return ! Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
	}

	/**
	 * Get the selector for the animation type 'Item by item'.
	 *
	 * @since 1.20.0
	 */
	public function get_separate_animation_selector() {
		return '.elementor-widget-cmsmasters-advanced-title__item';
	}

	/**
	 * Get the class name ot the block for the text animation types 'Line by Line', 'Word by Word', 'Char by Char', etc.
	 * and names of animation types suitable for this widget.
	 *
	 * @since 1.20.0
	 */
	public function get_text_animation_class() {
		return array( 'elementor-widget-cmsmasters-advanced-title__item' => 'sequental, random, line, word, char' );
	}

	public function get_widget_class() {
		return 'elementor-widget-cmsmasters-advanced-title';
	}

	public function get_widget_selector() {
		return '.' . $this->get_widget_class();
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to the widget.
	 *
	 * Should be inherited and register new controls using `add_control()`,
	 * `add_responsive_control()` and `add_group_control()`, inside control
	 * wrappers like `start_controls_section()`, `start_controls_tabs()` and
	 * `start_controls_tab()`.
	 *
	 * @since 1.20.0
	 */
	protected function register_controls() {
		/* Content Tab */
		$this->register_content_controls_general();

		/* Style Tab */
		$this->register_style_controls_general();
	}

	/**
	 * Advanced Title content controls.
	 *
	 * @since 1.20.0
	 */
	protected function register_content_controls_general() {
		$this->start_controls_section(
			'section_advanced_title',
			array( 'label' => esc_html__( 'Advanced Title', 'cmsmasters-elementor' ) )
		);

		$this->add_control(
			'entrance_animation',
			array(
				'label' => esc_html__( 'Entrance Animation', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'entrance_animation_text',
			array(
				'label' => esc_html__( 'Entrance Animation Text', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'yes',
			)
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'advanced_title_type',
			array(
				'label' => esc_html__( 'Type', 'cmsmasters-elementor' ),
				'type' => CmsmastersControls::CHOOSE_TEXT,
				'options' => array(
					'text' => array(
						'title' => esc_html__( 'Text', 'cmsmasters-elementor' ),
					),
					'icon' => array(
						'title' => esc_html__( 'Icon', 'cmsmasters-elementor' ),
					),
				),
				'default' => 'text',
				'toggle' => false,
				'label_block' => false,
				'render_type' => 'template',
			)
		);

		$repeater->add_control(
			'advanced_title_text',
			array(
				'label' => esc_html__( 'Text', 'cmsmasters-elementor' ),
				'label_block' => true,
				'type' => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter your text here', 'cmsmasters-elementor' ),
				'default' => esc_html__( 'Enter your text here', 'cmsmasters-elementor' ),
				'condition' => array( 'advanced_title_type' => 'text' ),
			)
		);

		$repeater->add_control(
			'advanced_title_icon',
			array(
				'label' => esc_html__( 'Icon', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::ICONS,
				'default' => array(
					'value' => 'fas fa-star',
					'library' => 'fa-solid',
				),
				'recommended' => array(
					'fa-solid' => array(
						'star',
						'circle',
						'dot-circle',
						'square-full',
					),
					'fa-regular' => array(
						'circle',
						'dot-circle',
						'square-full',
					),
				),
				'condition' => array( 'advanced_title_type' => 'icon' ),
			)
		);

		$repeater->add_control(
			'advanced_title_icon_size',
			array(
				'label' => esc_html__( 'Size', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'selectors' => array(
					'{{WRAPPER}} {{CURRENT_ITEM}}' => '--cmsmasters-advanced-title-icon-size: {{SIZE}}{{UNIT}};',
				),
				'condition' => array( 'advanced_title_type' => 'icon' ),
			)
		);

		$repeater->add_control(
			'advanced_title_icon_position',
			array(
				'label' => esc_html__( 'Vertical Position', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'selectors' => array(
					'{{WRAPPER}} {{CURRENT_ITEM}}' => '--cmsmasters-advanced-title-icon-position: {{SIZE}}{{UNIT}};',
				),
				'condition' => array( 'advanced_title_type' => 'icon' ),
			)
		);

		$repeater->add_control(
			'advanced_title_item_link',
			array(
				'label' => esc_html__( 'Link', 'cmsmasters-elementor' ),
				'label_block' => true,
				'type' => Controls_Manager::URL,
				'dynamic' => array( 'active' => true ),
				'placeholder' => esc_html__( 'https://your-link.com', 'cmsmasters-elementor' ),
			)
		);

		$repeater->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name' => 'advanced_title_item_typography',
				'selector' => '{{WRAPPER}} {{CURRENT_ITEM}}.cmsmasters-advanced-title-item-text',
				'condition' => array( 'advanced_title_type' => 'text' ),
			)
		);

		$repeater->add_control(
			'advanced_title_color',
			array(
				'label' => esc_html__( 'Color', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} {{CURRENT_ITEM}}' => '--text-color: {{VALUE}}; --icon-color: {{VALUE}};',
					'{{WRAPPER}}.cmsmasters-color-variation-gradient {{CURRENT_ITEM}}' => '-webkit-text-fill-color: {{VALUE}};',
					'{{WRAPPER}}.cmsmasters-color-variation-background-image {{CURRENT_ITEM}}' => '-webkit-text-fill-color: {{VALUE}};',
				),
			)
		);

		$repeater->add_control(
			'advanced_title_color_hover',
			array(
				'label' => esc_html__( 'Hover Color', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} {{CURRENT_ITEM}}' => '--text-color-hover: {{VALUE}}; --icon-hover-color: {{VALUE}};',
					'{{WRAPPER}}.cmsmasters-color-variation-gradient {{CURRENT_ITEM}}:hover' => '-webkit-text-fill-color: {{VALUE}};',
					'{{WRAPPER}}.cmsmasters-color-variation-background-image {{CURRENT_ITEM}}:hover' => '-webkit-text-fill-color: {{VALUE}};',
					'{{WRAPPER}}.cmsmasters-color-variation-gradient .cmsmasters-global-link:hover {{CURRENT_ITEM}}' => '-webkit-text-fill-color: {{VALUE}};',
					'{{WRAPPER}}.cmsmasters-color-variation-background-image .cmsmasters-global-link:hover {{CURRENT_ITEM}}' => '-webkit-text-fill-color: {{VALUE}};',
				),
			)
		);

		$repeater->add_responsive_control(
			'margin',
			array(
				'label' => esc_html__( 'Margin', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => array(
					'px',
					'%',
					'em',
					'rem',
					'vw',
					'vh',
					'custom',
				),
				'allowed_dimensions' => 'horizontal',
				'placeholder' => array(
					'top' => '',
					'right' => 'auto',
					'bottom' => '',
					'left' => 'auto',
				),
				'selectors' => array(
					'{{WRAPPER}} {{CURRENT_ITEM}}' => "--cmsmasters-advanced-title-item-margin-top: {{TOP}}{{UNIT}}; --cmsmasters-advanced-title-item-margin-right: {{RIGHT}}{{UNIT}}; --cmsmasters-advanced-title-item-margin-bottom: {{BOTTOM}}{{UNIT}}; --cmsmasters-advanced-title-item-margin-left: {{LEFT}}{{UNIT}};",
				),
			)
		);

		$repeater->add_control(
			'advanced_title_layout',
			array(
				'label' => esc_html__( 'Start New Line', 'cmsmasters-elementor' ),
				'label_block' => false,
				'type' => Controls_Manager::SWITCHER,
				'default' => '',
				'return_value' => 'block',
				'render_type' => 'template',
			)
		);

		$this->add_control(
			'content',
			array(
				'label' => esc_html__( 'Content', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => array(
					array(
						'advanced_title_text' => esc_html__( 'Enter your text here', 'cmsmasters-elementor' ),
					),
				),
				'title_field' => '{{{ "text" === advanced_title_type ? advanced_title_text : advanced_title_type }}}',
				'render_type' => 'template',
			)
		);

		$this->add_control(
			'link',
			array(
				'label' => esc_html__( 'Link', 'cmsmasters-elementor' ),
				'label_block' => true,
				'type' => Controls_Manager::URL,
				'dynamic' => array( 'active' => true ),
				'placeholder' => esc_html__( 'https://your-link.com', 'cmsmasters-elementor' ),
			)
		);

		$this->add_control(
			'tag',
			array(
				'label' => esc_html__( 'HTML Tag', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => array(
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				),
				'default' => 'h2',
			)
		);

		$this->add_control(
			'advanced_title_blend_mode',
			array(
				'label' => esc_html__( 'Blend Mode', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => array(
					'' => esc_html__( 'Normal', 'cmsmasters-elementor' ),
					'multiply' => 'Multiply',
					'screen' => 'Screen',
					'overlay' => 'Overlay',
					'darken' => 'Darken',
					'lighten' => 'Lighten',
					'color-dodge' => 'Color Dodge',
					'saturation' => 'Saturation',
					'color' => 'Color',
					'difference' => 'Difference',
					'exclusion' => 'Exclusion',
					'hue' => 'Hue',
					'luminosity' => 'Luminosity',
				),
				'selectors' => array(
					'{{WRAPPER}}' => '--cmsmasters-advanced-title-mix-blend-mode: {{VALUE}};',
				),
				'separator' => 'before',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Register Advanced Title controls in Style tab.
	 *
	 * @since 1.20.0
	 */
	protected function register_style_controls_general() {
		$this->start_controls_section(
			'section_advanced_title_style',
			array(
				'label' => esc_html__( 'Title', 'cmsmasters-elementor' ),
				'tab' => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'advanced_title_alignment',
			array(
				'label' => esc_html__( 'Alignment', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => array(
					'left' => array(
						'title' => esc_html__( 'Left', 'cmsmasters-elementor' ),
						'icon' => 'eicon-text-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'cmsmasters-elementor' ),
						'icon' => 'eicon-text-align-center',
					),
					'right' => array(
						'title' => esc_html__( 'Right', 'cmsmasters-elementor' ),
						'icon' => 'eicon-text-align-right',
					),
				),
				'default' => 'left',
				'selectors' => array(
					'{{WRAPPER}}' => '--cmsmasters-advanced-title-text-align: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name' => 'advanced_title_global_typography',
				'selector' => '{{WRAPPER}} .elementor-widget-cmsmasters-advanced-title__title',
			)
		);

		$this->start_controls_tabs( 'advanced_title_style_tabs' );

		$this->start_controls_tab(
			'advanced_title_style_tabs_normal_tab',
			array( 'label' => esc_html__( 'Normal', 'cmsmasters-elementor' ) )
		);

		$this->add_group_control(
			Group_Control_Vars_Text_Background::get_type(),
			array(
				'name' => 'advanced_title_bg',
				'selector' => '{{WRAPPER}}',
			)
		);

		$this->add_control(
			'advanced_title_style_icon_color_normal',
			array(
				'label' => esc_html__( 'Icon Color', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}}' => '--icon-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'advanced_title_style_text_stroke_width',
			array(
				'label' => esc_html__( 'Stroke Width', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => array(
					'px',
					'em',
				),
				'range' => array(
					'px' => array(
						'min' => 0,
						'max' => 20,
					),
					'em' => array(
						'min' => 0,
						'max' => 0.2,
						'step' => 0.01,
					),
				),
				'default' => array(
					'unit' => 'px',
					'size' => '',
				),
				'selectors' => array(
					'{{WRAPPER}}' => '--cmsmasters-advanced-title-text-stroke-width: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'advanced_title_style_text_stroke_color',
			array(
				'label' => esc_html__( 'Stroke Color', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}}' => '--cmsmasters-advanced-title-text-stroke-color: {{VALUE}};',
				),
				'conditions' => array(
					'relation' => 'and',
					'terms' => array(
						array(
							'name' => 'advanced_title_style_text_stroke_width[size]',
							'operator' => '>',
							'value' => '0',
						),
					),
				),
			)
		);

		$this->add_group_control(
			CmsmastersControls::VARS_TEXT_SHADOW_GROUP,
			array( 'name' => 'advanced_title_style' )
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'advanced_title_style_tabs_hover_tab',
			array( 'label' => esc_html__( 'Hover', 'cmsmasters-elementor' ) )
		);

		$this->add_control(
			'advanced_title_bg_hover_text_color',
			array(
				'label' => esc_html__( 'Color', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}}' => '--text-color-hover: {{VALUE}};',
				),
				'condition' => array( 'advanced_title_bg_text_bg_variation' => 'default' ),
			)
		);

		$this->add_control(
			'advanced_title_style_icon_color_hover',
			array(
				'label' => esc_html__( 'Icon Color', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}}' => '--icon-hover-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'advanced_title_style_text_stroke_width_hover',
			array(
				'label' => esc_html__( 'Stroke Width', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => array(
					'px',
					'em',
				),
				'range' => array(
					'px' => array(
						'min' => 0,
						'max' => 20,
					),
					'em' => array(
						'min' => 0,
						'max' => 0.2,
						'step' => 0.01,
					),
				),
				'default' => array(
					'unit' => 'px',
					'size' => '',
				),
				'selectors' => array(
					'{{WRAPPER}}' => '--cmsmasters-advanced-title-text-stroke-width-hover: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'advanced_title_style_text_stroke_color_hover',
			array(
				'label' => esc_html__( 'Stroke Color', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}}' => '--cmsmasters-advanced-title-text-stroke-color-hover: {{VALUE}};',
				),
				'conditions' => array(
					'relation' => 'and',
					'terms' => array(
						array(
							'name' => 'advanced_title_style_text_stroke_width_hover[size]',
							'operator' => '>',
							'value' => '0',
						),
					),
				),
			)
		);

		$this->add_group_control(
			CmsmastersControls::VARS_TEXT_SHADOW_GROUP,
			array( 'name' => 'advanced_title_style_hover' )
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'advanced_title_style_space_between',
			array(
				'label' => esc_html__( 'Space Between', 'cmsmasters-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => array(
					'px',
					'em',
					'vw',
					'vh',
					'custom',
				),
				'range' => array(
					'px' => array(
						'min' => 0,
						'max' => 200,
					),
				),
				'separator' => 'before',
				'selectors' => array(
					'{{WRAPPER}}' => '--cmsmasters-advanced-title-space-between: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render advanced title widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.20.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_render_attribute( '_wrapper', array(
			'data-separate-animation-selector' => $this->get_separate_animation_selector(),
			'data-text-animation-class' => $this->get_text_animation_class(),
		) );

		$this->add_render_attribute( 'advanced-title', 'class', array( $this->get_widget_class() . '__title' ) );

		if ( ! empty( $settings['link']['url'] ) ) {
			$this->add_render_attribute( 'advanced-title', 'class', 'cmsmasters-global-link' );
		}

		echo '<' . esc_html( $settings['tag'] ) . ' ' . $this->get_render_attribute_string( 'advanced-title' ) . '>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

		$idx = 0;

		foreach ( $settings['content'] as $item ) {
			$idx++;

			$link = array();
			$link_url = '';

			if ( ! empty( $settings['link']['url'] ) ) {
				$link = $settings['link'];
				$link_url = $settings['link']['url'];
			} elseif ( empty( $settings['link']['url'] ) && ! empty( $item['advanced_title_item_link']['url'] ) ) {
				$link = $item['advanced_title_item_link'];
				$link_url = $item['advanced_title_item_link']['url'];
			}

			$this->add_link_attributes( 'item-link-' . $idx, $link );

			$start_tag = $link_url ? '<a ' . wp_kses_post( $this->get_render_attribute_string( 'item-link-' . $idx ) ) : '<span';
			$separator = ( 'block' === $item['advanced_title_layout'] ? '<br />' : '' );
			$end_tag = $link_url ? '</a>' : '</span>';

			$this->add_render_attribute( 'advanced-title-item' . $idx, 'class', array(
				$this->get_widget_class() . '__item',
				'elementor-repeater-item-' . esc_attr( $item['_id'] ),
			) );

			switch ( $item['advanced_title_type'] ) {
				case 'text':
					$this->add_render_attribute( 'advanced-title-item' . $idx, 'class', 'cmsmasters-advanced-title-item-text' );

					echo $separator; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

					echo $start_tag . ' ' . $this->get_render_attribute_string( 'advanced-title-item' . $idx ) . '>' . // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						'<span class="' . $this->get_widget_class() . '__text-wrap">' . // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
							'<span class="' . $this->get_widget_class() . '__text">' . // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
								esc_html( $item['advanced_title_text'] ) .
							'</span>' .
						'</span>' .
					$end_tag; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

					break;
				case 'icon':
					ob_start();

					if ( '' !== $item['advanced_title_icon']['value'] ) {
						Icons_Manager::render_icon( $item['advanced_title_icon'], array(
							'aria-hidden' => 'true',
							'aria-label' => esc_attr__( 'Advanced Title item icon', 'cmsmasters-elementor' ),
							'class' => 'char',
						) );
					} else {
						Icons_Manager::render_icon(
							array(
								'value' => 'fas fa-circle',
								'library' => 'fa-solid',
							),
							array(
								'aria-hidden' => 'true',
								'aria-label' => esc_attr__( 'Advanced Title item icon', 'cmsmasters-elementor' ),
								'class' => 'char',
							)
						);
					}

					$icon = ob_get_clean();

					if ( false !== strpos( $icon, '<svg' ) ) {
						$icon = '<span class="' . $this->get_widget_class() . '__item-icon-svg">' .
							$icon .
						'</span>';
					}

					$this->add_render_attribute( 'advanced-title-item' . $idx, 'class', 'cmsmasters-advanced-title-item-icon' );

					echo $separator; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

					echo $start_tag . ' ' . $this->get_render_attribute_string( 'advanced-title-item' . $idx ) . '>' . // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						$icon . // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
					$end_tag; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

					break;
				default:
					break;
			}
		}

		echo '</' . esc_html( $settings['tag'] ) . '>';
	}

	/**
	 * Render a widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @access protected
	 */
	protected function content_template() {
		?>
		<# var globalLink = ( settings.link && settings.link.url ) ? ' cmsmasters-global-link' : ''; #>

		<{{{ settings.tag }}} class="elementor-widget-cmsmasters-advanced-title__title {{{ globalLink }}}" {{{ view.getRenderAttributeString( 'advanced-title' ) }}}><#

		settings.content.forEach( function( item ) {
			var link_url = '',
				link_target = '',
				link_nofollow = '';

			if ( settings.link.url ) {
				link_url = settings.link.url;
				link_target = settings.link.is_external ? ' target="_blank"' : '';
				link_nofollow = settings.link.nofollow ? ' rel="nofollow"' : '';
			} else if ( ! settings.link.url && item.advanced_title_item_link.url != '' ) {
				link_url = item.advanced_title_item_link.url;
				link_target = item.advanced_title_item_link.is_external ? ' target="_blank"' : '';
				link_nofollow = item.advanced_title_item_link.nofollow ? ' rel="nofollow"' : '';
			}

			var start_tag = link_url ? '<a href="' + link_url + '"' + link_target + link_nofollow : '<span';
			var end_tag = link_url ? '</a>' : '</span>';
			var separator = ( item.advanced_title_layout == 'block' ? '<br />' : '' );

			switch ( item.advanced_title_type ) {
				case 'text':
					#>{{{ separator }}}{{{ start_tag }}} class="elementor-widget-cmsmasters-advanced-title__item cmsmasters-advanced-title-item-text elementor-repeater-item-{{{ item._id }}}"><#
						#><span class="elementor-widget-cmsmasters-advanced-title__text-wrap"><#
							#><span class="elementor-widget-cmsmasters-advanced-title__text">{{{ item.advanced_title_text }}}</span><#
						#></span><#
					#>{{{ end_tag }}}<#
					break;
				case 'icon':
					var iconHTML = elementor.helpers.renderIcon(
						view,
						item.advanced_title_icon,
						{ 'aria-hidden': true, 'aria-label': 'Advanced Title item icon', 'class': 'char' },
						'i',
						'value'
					);

					if ( '' !== item.advanced_title_icon.value ) {
						if ( 'object' === typeof iconHTML && iconHTML.rendered ) {
							iconHTML = iconHTML.value;
						}

						if ( iconHTML && iconHTML.indexOf( '<svg' ) >= 0 ) {
							iconHTML = '<span class="elementor-widget-cmsmasters-advanced-title__item-icon-svg">' + iconHTML + '</span>';
						}
					} else {
						#><span class="elementor-widget-cmsmasters-advanced-title__item-icon-svg">
							<i aria-hidden="true" aria-label="Advanced Title item icon" class="fas fa-circle"></i>
						</span><#
					}

					#>{{{ separator }}}{{{ start_tag }}} class="elementor-widget-cmsmasters-advanced-title__item cmsmasters-advanced-title-item-icon elementor-repeater-item-{{{ item._id }}}">{{{ iconHTML }}}{{{ end_tag }}}<#

					break;
				default:
					break;
			}
		} );

		#></{{{ settings.tag }}}><?php
	}
}
